/*
  Macro           AnalogClock
  Author          Carlo Hogeveen
  Website         eCarlo.nl/tse
  Compatibility   Windows GUI and Console TSE v4 upwards
  Version         v0.0.0.3   30 Jan 2026

  The idea is to have a small analog clock in a corner of the editor.

  Currently this development version displays a huge running flickering clock
  until you press a key.

  In Windows Console TSE that clock face is not sized correctly and gets
  partially overwritten around the cursor line.


  INSTALLATION AND USAGE

  For now, just compile the macro and execute it.


  TODO
    Unflickering the clock, clock sizing and positioning, clock updating, bugs.


  HISTORY

  v0.0.0.3   30 Jan 2026
    Displays a huge running flickering clock until you press a key.

  v0.0.0.2   29 Jan 2026
    Shows main and small clock-hands and stops.

  v0.0.0.1   26 Jan 2026
    Initial dev version. Just shows huge clock-face and hour-marks.
*/



/*
  T E C H N I C A L   B A C K G R O U N D   I N F O


  WINDOWS FUNCTIONS

    https://learn.microsoft.com/en-us/windows/win32/gdi/filled-shape-functions
    https://learn.microsoft.com/en-us/windows/win32/gdi/line-and-curve-functions


  MATH

    I was both dreading and exited about having to write a sine and cosine
    function, but this macro only uses angles of whole degrees, for which
    a table of precalculated sine/cosine results is sufficient.

    Because TSE's macro language cannot handle decimals, the macro works
    with "mega-values" where necessary.
    A mega-value is a million-fold of the actual value.

    This allows the use of sine and cosine results with 6 digits of precision.

    Combined with TSE's integer size this limits the vertical and horizontal
    window-size in pixels that the macro can handle, namely to up to 10000 in
    width and height.
    Research in Jan 2026 did not find consumer monitors where the shortest of
    the width and height in pixels exceeds 10000.
    Just to be sure the macro gives an error message when this happens.
*/



// Start of compatibility restrictions and mitigations ...

#ifdef LINUX
  This macro is not compatible with LINUX TSE.
#endif

#ifdef WIN32
#else
   16-bit versions of TSE are not supported. You need at least TSE v4.
#endif

#ifdef EDITOR_VERSION
#else
   Editor Version is older than TSE v3. You need at least TSE v4.
#endif

#if EDITOR_VERSION < 4000h
   Editor Version is too old. You need at least TSE v4.
#endif

// End of compatibility restrictions and mitigations.



//  Constants

//  Macro constants and semi-constants
#define CLR_INVALID                 0xFFFFFFFF
string  MACRO_NAME [MAXSTRINGLEN] = ''
#define MEGA                        1000000
#define NULL                        0
#define ROOT_MAXINT                 46340

//  Windows brushes
#define WHITE_BRUSH   0
#define LTGRAY_BRUSH  1
#define GRAY_BRUSH    2
#define DKGRAY_BRUSH  3
#define BLACK_BRUSH   4
#define NULL_BRUSH    5
#define HOLLOW_BRUSH  NULL_BRUSH

//  Windows pens
#define WHITE_PEN     6
#define BLACK_PEN     7
#define NULL_PEN      8



// Start of Windows DLL functions

dll "<user32.dll>"

  integer proc IsMaximized(integer hwnd) : "IsZoomed"

  integer proc GetDC(integer hWnd)
  integer proc ReleaseDC(integer hWnd, integer hDC)

  integer proc GetClientRect(integer hWnd, integer lpRect)
  integer proc GetWindowRect(integer hWnd, integer lpRect)

  integer proc HideCaret(integer hWnd)
  integer proc ShowCaret(integer hWnd)

end


dll "<gdi32.dll>"

  integer proc CreateSolidBrush(integer bgr_color)
  integer proc DeleteObject(integer h)
  integer proc Ellipse(integer hdc, integer x_left, integer y_top,
                       integer x_right, integer y_bottom)
  integer proc GetPixel(integer hdc, integer x_pos, integer y_pos)//Returns BGR.
  integer proc GetStockObject(integer i)
  integer proc Rectangle(integer hdc, integer x_left, integer y_top,
                         integer x_right, integer y_bottom)
  integer proc SelectObject(integer hdc, integer h)
  integer proc SetPixel(integer hdc, integer x_pos, integer y_pos,
                        integer bgr_color)

  integer proc Polygon(integer hdc, integer apt, integer cpt)

  integer proc MoveToEx(integer hdc, integer x, integer y, integer lppt)
  integer proc LineTo(integer hdc, integer x, integer y)


  //  These procs are unused, because I did not get them to work,
  //  and it turns out I did not need them (yet).
  integer proc GetDCBrushColor(integer hdc)
  integer proc SetDCBrushColor(integer hdc, integer bgr_color)
  integer proc GetDCPenColor(integer hdc)
  integer proc SetDCPenColor(integer hdc, integer bgr_color)

end

// End of Windows DLL functions



Datadef mega_sines_per_degree
  "0        0"
  "1    17452"
  "2    34899"
  "3    52336"
  "4    69756"
  "5    87156"
  "6   104528"
  "7   121869"
  "8   139173"
  "9   156434"
  "10  173648"
  "11  190809"
  "12  207912"
  "13  224951"
  "14  241922"
  "15  258819"
  "16  275637"
  "17  292372"
  "18  309017"
  "19  325568"
  "20  342020"
  "21  358368"
  "22  374607"
  "23  390731"
  "24  406737"
  "25  422618"
  "26  438371"
  "27  453990"
  "28  469472"
  "29  484810"
  "30  500000"
  "31  515038"
  "32  529919"
  "33  544639"
  "34  559193"
  "35  573576"
  "36  587785"
  "37  601815"
  "38  615661"
  "39  629320"
  "40  642788"
  "41  656059"
  "42  669131"
  "43  681998"
  "44  694658"
  "45  707107"
  "46  719340"
  "47  731354"
  "48  743145"
  "49  754710"
  "50  766044"
  "51  777146"
  "52  788011"
  "53  798636"
  "54  809017"
  "55  819152"
  "56  829038"
  "57  838671"
  "58  848048"
  "59  857167"
  "60  866025"
  "61  874620"
  "62  882948"
  "63  891007"
  "64  898794"
  "65  906308"
  "66  913545"
  "67  920505"
  "68  927184"
  "69  933580"
  "70  939693"
  "71  945519"
  "72  951057"
  "73  956305"
  "74  961262"
  "75  965926"
  "76  970296"
  "77  974370"
  "78  978148"
  "79  981627"
  "80  984808"
  "81  987688"
  "82  990268"
  "83  992546"
  "84  994522"
  "85  996195"
  "86  997564"
  "87  998630"
  "88  999391"
  "89  999848"
  "90 1000000"
end mega_sines_per_degree


integer g_mega_sines_per_degree_id = 0


//  Returns sin(degrees) * 1,000,000.
integer proc mega_sin(integer degrees)
  integer lookup_degrees = ((degrees mod 360) + 360) mod 360
  integer result         = 0
  integer sign           = 1

  PushLocation()

  if g_mega_sines_per_degree_id
    GotoBufferId(g_mega_sines_per_degree_id)
  else
    g_mega_sines_per_degree_id = CreateTempBuffer()
    ChangeCurrFilename(SplitPath(CurrMacroFilename(), _NAME_) +
                         ':MegaSinesPerDegree',
                       _DONT_PROMPT_|_DONT_EXPAND_|_OVERWRITE_)
    InsertData(mega_sines_per_degree)
    UnMarkBlock()
  endif

  //  Given that looking up a value in the table relatively takes a lot of
  //  run-time, one way to improve that it is to keep the table small by
  //  omitting its duplicate values and adjusting the lookup for that.

  case lookup_degrees
    when 91 .. 180
      lookup_degrees = 180 - lookup_degrees
    when 181 .. 270
      lookup_degrees = lookup_degrees - 180
      sign           = -1
    when 271 .. 360
      lookup_degrees = 360 - lookup_degrees
      sign           = -1
  endcase

  if lFind(Str(lookup_degrees) + ' ', '^g')
    result = sign * Val(GetToken(GetText(1, MAXSTRINGLEN), ' ', 2))
  endif

  PopLocation()
  return(result)
end mega_sin


//  Returns cos(degrees) * 1,000,000.
integer proc mega_cos(integer degrees)
  return(mega_sin(degrees + 90))
end mega_cos


proc draw_polygon(integer device_context_handle,
                  string  coordinates,
                  integer bgr_color)
  integer array_of_coordinates                       = 0
  string  array_of_coordinates_string [MAXSTRINGLEN] = ''
  integer color_brush                                = 0
  integer coordinate_counter                         = 0
  integer coordinate_addr                            = 0
  integer coordinate_value                           = 0
  integer old_brush                                  = 0
  integer old_pen                                    = 0

  //  Both draw_polygon() and Polygon() interpret a sequence of coordinates
  //  as a sequence of pairs of coordinates that describe a point.

  array_of_coordinates = AdjPtr(Addr(array_of_coordinates_string), 2)

  for coordinate_counter = 1 to Min(NumTokens(coordinates, ' '), 62)
    coordinate_value = Val(GetToken(coordinates, ' ', coordinate_counter))
    coordinate_addr  = AdjPtr(array_of_coordinates, (coordinate_counter - 1) * 4)
    PokeLong(coordinate_addr, coordinate_value)
  endfor

  // Create a BGR-colored brush
  // HBRUSH hGreen = CreateSolidBrush(RGB(0, 255, 0));
  color_brush = CreateSolidBrush(bgr_color)

  // Select it into the DC, replacing the default brush
  // HBRUSH hOldBrush = (HBRUSH)SelectObject(hdc, hGreen);
  old_brush = SelectObject(device_context_handle, color_brush)

  // Use a stock pen or NULL_PEN if you don't want a border
  // HPEN hOldPen = (HPEN)SelectObject(hdc, GetStockObject(BLACK_PEN));
  old_pen = SelectObject(device_context_handle, GetStockObject(NULL_PEN)) // No border.

  polygon(device_context_handle, array_of_coordinates, coordinate_counter / 2)

  // Restore the original objects
  SelectObject(device_context_handle, old_brush)
  SelectObject(device_context_handle, old_pen)

  // Delete the brush we created
  DeleteObject(color_brush)
end draw_polygon


proc draw_ellipse(integer device_context_handle, integer x_left, integer y_top,
                  integer x_right, integer y_bottom, integer bgr_color)
  // Create a BGR-colored brush
  // HBRUSH hGreen = CreateSolidBrush(RGB(0, 255, 0));
  integer color_brush = CreateSolidBrush(bgr_color)

  // Select it into the DC, replacing the default brush
  // HBRUSH hOldBrush = (HBRUSH)SelectObject(hdc, hGreen);
  integer old_brush = SelectObject(device_context_handle, color_brush)

  // Use a stock pen or NULL_PEN if you don't want a border
  // HPEN hOldPen = (HPEN)SelectObject(hdc, GetStockObject(BLACK_PEN));
  integer old_pen = SelectObject(device_context_handle, GetStockObject(NULL_PEN)) // No border.

  // Draw the rectangle
  ellipse(device_context_handle, x_left, y_top, x_right, y_bottom)

  // Restore the original objects
  SelectObject(device_context_handle, old_brush)
  SelectObject(device_context_handle, old_pen)

  // Delete the brush we created
  DeleteObject(color_brush)
end draw_ellipse


proc draw_circle(integer device_context_handle,
                 integer center_x,
                 integer center_y,
                 integer radius,
                 integer bgr_color)
  draw_ellipse(device_context_handle,
               center_x - radius,
               center_y - radius,
               center_x + radius,
               center_y + radius,
               bgr_color)
end draw_circle


/*
proc draw_rectangle(integer hdc, integer x_left, integer y_top,
                    integer x_right, integer y_bottom, integer bgr_color)
  // Create a green brush
  // HBRUSH hGreen = CreateSolidBrush(RGB(0, 255, 0));
  integer color_brush = CreateSolidBrush(bgr_color)

  // Select it into the DC, replacing the default brush
  // HBRUSH hOldBrush = (HBRUSH)SelectObject(hdc, hGreen);
  integer old_brush = SelectObject(hdc, color_brush)

  // Use a stock pen or NULL_PEN if you don't want a border
  // HPEN hOldPen = (HPEN)SelectObject(hdc, GetStockObject(BLACK_PEN));
  integer old_pen = SelectObject(hdc, GetStockObject(NULL_PEN)) // No border.

  // Draw the rectangle
  Rectangle(hdc, x_left, y_top, x_right, y_bottom)

  // Restore the original objects
  SelectObject(hdc, old_brush)
  SelectObject(hdc, old_pen)

  // Delete the brush we created
  DeleteObject(color_brush)
end draw_rectangle
*/


proc get_window_pixel_borders(    integer win_handle,
                              var integer win_left,
                              var integer win_top,
                              var integer win_right,
                              var integer win_bottom)
  string  win_rect [16] = Format('':16:' ')
  integer win_rect_addr = AdjPtr(Addr(win_rect), 2)

  GetClientRect(win_handle, win_rect_addr)

  win_left   = PeekLong(       win_rect_addr     )
  win_top    = PeekLong(AdjPtr(win_rect_addr,  4))
  win_right  = PeekLong(AdjPtr(win_rect_addr,  8))
  win_bottom = PeekLong(AdjPtr(win_rect_addr, 12))
end get_window_pixel_borders


proc analog_clock()
  integer clock_diameter        = 0
  integer clock_radius          = 0
  integer clock_center_x        = 0
  integer clock_center_y        = 0
  integer degrees               = 0
  integer device_context_handle = 0
  integer hours_times_100       = 0
  integer mark_diameter         = 0
  integer mark_radius           = 0
  integer mark_center_x         = 0
  integer mark_center_y         = 0
  integer minutes               = 0
  integer seconds               = 0
  integer win_bottom            = 0
  integer win_handle            = 0
  integer win_left              = 0
  integer win_right             = 0
  integer win_top               = 0

  win_handle            = GetWinHandle()
  device_context_handle = GetDC(win_handle)

  get_window_pixel_borders(win_handle, win_left, win_top, win_right, win_bottom)

  clock_diameter = Min(win_right - win_left, win_bottom - win_top) - 200
  clock_radius   = clock_diameter / 2

  if clock_diameter > 10000
    MsgBox(MACRO_NAME,
           'Aborted: TSE window with short side > 10000 pixels not supported.')
  else
    //  Draw the clock's face as a big circle.
    clock_center_x = clock_radius
    clock_center_y = clock_radius
    draw_circle(device_context_handle, clock_center_x, clock_center_y,
                clock_radius, 0xffffff)   //  White.

    //  Draw 12 hour-marks as small circles that touch the big face circle.
    mark_diameter = clock_diameter / 10
    mark_radius   = clock_diameter / 20
    for degrees = 0 to 360 - 1 by 360 / 12
      mark_center_x = clock_center_x +
                      mega_cos(degrees) * (clock_radius - mark_radius) / MEGA
      mark_center_y = clock_center_y +
                      mega_sin(degrees) * (clock_radius - mark_radius) / MEGA
      draw_circle(device_context_handle, mark_center_x, mark_center_y,
                  mark_radius, 0x000000)  //  Black.
    endfor

    //  Draw a similar center-mark at the center of the clock-face.
    draw_circle(device_context_handle, clock_center_x, clock_center_y,
                mark_radius, 0x000000)  //  Black.

    //  Draw main hand as triangle from center-mark tapering to hour-mark.
    minutes = GetTime() / 6000 mod 60
    degrees = 90 - minutes * 6  //  The directon of the main hand.

    //  Beware, the windows y coordinate grows downward and math's y coordinate
    //  grows upward, so we need to vertically mirror/subtract mega_sin().
    draw_polygon(device_context_handle,
                 Format(clock_center_x +
                        mega_cos(degrees - 90) * mark_radius / MEGA;
                        clock_center_y -
                        mega_sin(degrees - 90) * mark_radius / MEGA;

                        clock_center_x +
                        mega_cos(degrees + 90) * mark_radius / MEGA;
                        clock_center_y -
                        mega_sin(degrees + 90) * mark_radius / MEGA;

                        clock_center_x +
                        mega_cos(degrees     ) * (clock_radius - mark_diameter) / MEGA;
                        clock_center_y -
                        mega_sin(degrees     ) * (clock_radius - mark_diameter) / MEGA),
                 0x000000)  // Black.

    //  Draw small hand as triangle from center-mark tapering to hour-mark.
    //  Use hours_times_100 to proportionally draw small hand between whole hours.
    hours_times_100 = GetTime() / 3600 mod 1200
    degrees         = 90 - hours_times_100 * 360 / 1200  //  The directon of the small hand.

    //  Beware, the windows y coordinate grows downward and math's y coordinate
    //  grows upward, so we need to vertically mirror/subtract mega_sin().
    draw_polygon(device_context_handle,
                 Format(clock_center_x +
                        mega_cos(degrees - 90) * mark_radius / MEGA;
                        clock_center_y -
                        mega_sin(degrees - 90) * mark_radius / MEGA;

                        clock_center_x +
                        mega_cos(degrees + 90) * mark_radius / MEGA;
                        clock_center_y -
                        mega_sin(degrees + 90) * mark_radius / MEGA;

                        clock_center_x +
                        mega_cos(degrees     ) * (clock_radius - mark_diameter) * 4 / 5 / MEGA;
                        clock_center_y -
                        mega_sin(degrees     ) * (clock_radius - mark_diameter) * 4 / 5 / MEGA),
                 0x000000)  // Black.

    //  Draw the seconds hand as a single line from the clock's center.
    //  We use this different technique just to try it out.
    seconds = GetTime() / 100 mod 60
    degrees = 90 - seconds * 6  //  The directon of the seconds hand.

    MoveToEx(device_context_handle, clock_center_x, clock_center_y, NULL)
    //  Beware, the windows y coordinate grows downward and math's y coordinate
    //  grows upward, so we need to vertically mirror/subtract mega_sin().
    LineTo(device_context_handle,
           clock_center_x + mega_cos(degrees) * clock_radius / MEGA,
           clock_center_y - mega_sin(degrees) * clock_radius / MEGA)
  endif

  ReleaseDC(win_handle, device_context_handle)
end analog_clock


proc WhenLoaded()
  MACRO_NAME = SplitPath(CurrMacroFilename(), _NAME_)
end WhenLoaded


proc WhenPurged()
end WhenPurged


proc Main()
  while not KeyPressed()
    analog_clock()
    Delay(18)
  endwhile

  GetKey()

  //  UpdateDisplay(_ALL_WINDOWS_REFRESH_)does not refresh the whole screen.
  //  This trick does refresh the whole screen.
  PushKey(<Escape>)
  Help()


  PurgeMacro(CurrMacroFilename())
end Main

