/*
  Tool & extension    BasicColors
  Author              Carlo Hogeveen
  Website             eCarlo.nl/tse
  Compatibility       Windows GUI TSE v4.2 upwards, Linux TSE v4.50rc14 upwards
  Version             v1.1   24 Nov 2025 (2)


  I finally investigated an itch: Lately TSE's green felt too green.
  Upon investigation almost all colors turned out to be off from before.
  Testing shows that TSE v4.50.11 and older are not affected.

  This tool & extension can diagnose and repair the perceived problem,
  that TSE's colors are no longer its old basic colors as documented
  in TSE's Help topic for GetColorTableValue().

  If you execute this macro without only loading it first, then it will visually
  and textually report any deviations from these documented basic colors.
  Textually it will not report any colors that are unchanged.

  If you only load this macro, then it will set TSE's colors back to their
  basic values until TSE is restarted.

  Autoloading the macro will always give TSE its old basic colors.


  BE AWARE

    After only (auto)loading this macro, all colors are always basic colors.
    So executing the macro after that will not find any deviations.

    To see deviations again, you will need to remove this macro
    from the Macro AutoLoad List, restart TSE, and then execute the macro.


  HISTORY

  v1.1   24 Nov 2025 (2)
    Added a screen that visually shows any deviations from the basic colors.
    Made the macro do nothing in Windows Console TSE.

  v1     24 Nov 2025
    Initial release.

*/


// Global constants and semi-constants
string  BASIC_COLOR_VALUES [112] = '000000 000080 008000 008080 800000 800080 808000 c0c0c0 808080 0000ff 00ff00 00ffff ff0000  ff00ff ffff00 ffffff'
string  COLOR_NAMES        [160] = '_BLACK_  _BLUE_ _GREEN_ _CYAN_ _RED_ _MAGENTA_ _BROWN_ _LIGHT_GRAY_ _DARK_GRAY_ _LIGHT_BLUE_ _LIGHT_GREEN_ _LIGHT_CYAN_ _LIGHT_RED_ _LIGHT_MAG_ _YELLOW_ _WHITE_'
integer MACRO_ID                 = 0
string  NBSP                 [1] = ''
integer TEXT_COLOR               = Color(Bright White ON Black)


proc idle()
  AbandonFile(MACRO_ID)
  PurgeMacro(CurrMacroFilename())
end idle


/*

  -------- Foreground Colors --------  -------- Background Colors --------
  1             15       25 27     35  1             15       25 27     35
  Color name    Basic     ? Currently  Color name    Basic     ? Currently
  _LIGHT_GREEN_ xxxxxxxxx . xxxxxxxxx  _LIGHT_GREEN_ xxxxxxxxx . xxxxxxxxx
  ...

*/


proc show_colored_screen()
  integer color_index   = 0
  integer ground        = 0
  integer ground_column = 0
  integer old_color     = 0
  integer old_cursor    = Set(Cursor, OFF)
  string  equality  [1] = ''

  Set(Attr, Color(Black ON Black))
  ClrScr()

  PutStrAttrXY( 1, 1, '-------- Foreground Colors --------', '', Color(Bright White ON Black))
  PutStrAttrXY(41, 1, '-------- Background Colors --------', '', Color(Bright White ON Black))
  PutStrAttrXY( 1, 2, 'Color name    Basic     ? Currently', '', Color(Bright White ON Black))
  PutStrAttrXY(41, 2, 'Color name    Basic     ? Currently', '', Color(Bright White ON Black))

  for ground = _FOREGROUND_ to _BACKGROUND_
    ground_column = (ground - 1) * 40 + 1
    for color_index = 0 to 15
      if Val(GetToken(BASIC_COLOR_VALUES, ' ', color_index + 1), 16) ==
         GetBufferInt(Format(ground, '-', color_index), MACRO_ID)
        equality = '='
      else
        equality = '~'
      endif
      PutStrAttrXY(ground_column,
                   3 + color_index,
                   Format(GetToken(COLOR_NAMES, ' ', color_index + 1):-13,
                          equality:12),
                   '',
                   TEXT_COLOR)
      old_color = GetColorTableValue(_BACKGROUND_, _BLACK_)
      SetColorTableValue(_BACKGROUND_,
                         _BLACK_,
                         Val(GetToken(BASIC_COLOR_VALUES, ' ', color_index + 1),
                            16))
      PutStrAttrXY(ground_column + 14,
                   3 + color_index,
                   Format('':9:NBSP),
                   '',
                   TEXT_COLOR)
      SetColorTableValue(_BACKGROUND_,
                         _BLACK_,
                         GetBufferInt(Format(ground, '-', color_index),
                                      MACRO_ID))
      PutStrAttrXY(ground_column + 26,
                   3 + color_index,
                   Format('':9:NBSP),
                   '',
                   TEXT_COLOR)
      SetColorTableValue(_BACKGROUND_, _BLACK_, old_color)
    endfor
  endfor

  PutStrAttrXY(1, 20, 'Press any key ...', '', TEXT_COLOR)
  GetKey()
  Set(Cursor, old_cursor)
  UpdateDisplay(_ALL_WINDOWS_REFRESH_)
end show_colored_screen


proc WhenLoaded()
  integer color_index              = 0
  integer font_flags               = 0
  string  font_name [MAXSTRINGLEN] = ''
  integer font_size                = 0
  integer ground                   = 0

  #ifndef LINUX
    if not isGUI()
      PurgeMacro(CurrMacroFilename())
      return()
      Warn('Error: Not returned from WhenLoaded()!')
    endif
  #endif

  PushLocation()
  MACRO_ID = CreateTempBuffer()
  PopLocation()

  GetFont(font_name, font_size, font_flags)
  NBSP = iif(EquiStr(font_name, 'Terminal'), Chr(255), Chr(160))

  Hook(_IDLE_, idle)

  for ground = _FOREGROUND_ to _BACKGROUND_
    for color_index = 0 to 15
      SetBufferInt(Format(ground, '-', color_index),
                   GetColorTableValue(ground, color_index),
                   MACRO_ID)
      SetColorTableValue(ground, color_index,
                         Val(GetToken(BASIC_COLOR_VALUES,
                                      ' ',
                                      color_index + 1),
                             16))
    endfor
  endfor
end WhenLoaded


proc Main()
  integer color_index = 0
  integer ground      = 0
  integer org_id      = GetBufferId()

  #ifndef LINUX
    if not isGUI()
      PurgeMacro(CurrMacroFilename())
      return()
      Warn('Error: Not returned from Main()!')
    endif
  #endif

  for ground = _FOREGROUND_ to _BACKGROUND_
    for color_index = 0 to 15
      if GetColorTableValue(ground, color_index) <>
         GetBufferInt(Format(ground, '-', color_index), MACRO_ID)
        if GetBufferId() == org_id
          NewFile()
        endif
        AddLine(Format(iif(ground == _FOREGROUND_, 'Foreground', 'Background');
                       'color';
                       GetToken(COLOR_NAMES, ' ', color_index + 1):13;
                       'is';
                       GetBufferInt(Format(ground, '-', color_index),
                                    MACRO_ID):6:'0':16;
                       'instead of basic color';
                       GetColorTableValue(ground, color_index):6:'0':16,
                       '.'))
        SetColorTableValue(ground,
                           color_index,
                           GetBufferInt(Format(ground, '-', color_index),
                                        MACRO_ID))
      endif
    endfor
  endfor
  if GetBufferId() == org_id
    Warn("All colors are TSE's basic colors.")
  else
    show_colored_screen()
  endif
end Main


