/*
  Extension       BoxCursor
  Author          Carlo Hogeveen
  Website         eCarlo.nl/tse
  Compatibility   Windows GUI TSE
  Version         v0.0.0.7   24 Oct 2025 (2)

  This is an unfinished intermediate demo development version,
  that only very partially works, but demonstrates that drawing
  a box cursor with Windows APIs will work, though not "pretty" yet.
  Just compile and load it to see a box cursor.

  KNOWN DRAWBACK:
    It currently needs to set TSE's own cursor sizes to 0 to disable them.
    This unfortunately turns TSE's own cursor off permanently if you save TSE's
    settings.
    If this happens, then you can manually set TSE's cursor sizes again in its
    configuration menu Options -> Full Configuration -> Display/Color Options.

  This extension sets the cursor to an outlined box.
  meaning it draws a box with lines *between* characters.

  Disclaimer:
    This extension has not been rigorously tested yet.

  Compatibility:
    It is not compatible with Linux TSE.
    It is currently not compatible with Windows Console TSE.

  Installation:
    Compile it, and (auto)load it.


  TODO
    MUST
    - Finish making it a real cursor.
    - Make the default cursor color black or white
      based on the brightness of the text background color.
    SHOULD
    - Make the cursor speed configurable.
    - Make the cursor's color configurable.
    - Enable different cursor colors for insert mode and overwrite mode.
    COULD
    WONT


  HISTORY

  v0.0.0.7    24 Oct 2024 (2)
    Fixed: TSE's cursor is now restored if this macro is purged.

  v0.0.0.6    24 Oct 2024
   Disabled the TSE cursor. Fixed the drawback. Still need a steadier box
   cursor.

  v0.0.0.5    23 Oct 2024
    A simple box cursor now works, though not prettily yet, and with a drawback.

  v0.0.0.4    ?? Oct 2024
    Switching to Windows API's ...

  v0.0.0.3    17 Oct 2025
    Worked around box cursor making underscure invisible by never overwriting
    an underscore with the bottom line of the box cursor.

  v0.0.0.2    12 Oct 2025
    Unexpectedly was able to fix the screen flickering at each keystroke!

  v0.0.0.1    11 Oct 2025
    Initial version, which had two major known flaws.

*/




/*

  T E C H N I C A L   B A C K G R O U N D   I N F O


  (*1)  THE WEIRD "if EquiStr(MACRO_NAME ... " STATEMENTS

    Because Set(Cursor, ON/OFF) only works from an _IDLE_ hook,
    it is not simple to restore TSE's cursor when this macro is purged.
    The macro's own _IDLE_ hooks no longer get called after WhenPurged().
    The chosen solution is to spawn a renamed copy of itself,
    of which the only purpose is to reenable the cursor and then stop too.


  NO COLOR LIMITATION

    In theory this extension's drawing technique falls within TSE's limitations
    for giving each pixel its own RGB color. There are 16,777,216 RGB colors.


*/





//  Start of compatibility check ...

#ifdef LINUX
  This macro does not work in LINUX. It needs the Windows GUI Version of TSE.
#endif

//  End of compatibility check.





// Global constants and semi-constants.
string MACRO_NAME [MAXSTRINGLEN] = ''
string MACRO_DPN  [MAXSTRINGLEN] = ''



// Global variables
integer g_array_top               = 0
integer g_box_cursor_state        = OFF
integer g_device_context_handle   = 0
integer g_idle_counts             = 0
integer g_window_handle           = 0





// Start of Windows DLL APIs

dll "<user32.dll>"

  integer proc IsMaximized(integer hwnd) : "IsZoomed"

  integer proc GetDC(integer hWnd)
  integer proc ReleaseDC(integer hWnd, integer hDC)

  integer proc GetClientRect(integer hWnd, integer lpRect)
  integer proc GetWindowRect(integer hWnd, integer lpRect)

  integer proc HideCaret(integer hWnd)
  integer proc ShowCaret(integer hWnd)

end


dll "<gdi32.dll>"

  integer proc Rectangle(integer hdc, integer x_left, integer y_top,
                         integer x_right, integer y_bottom)

  integer proc SetPixel(integer hdc, integer x_pos, integer y_pos,
                        integer rgb_color)

  integer proc GetPixel(integer hdc, integer x_pos, integer y_pos) // Returns RGB.

end

// End of Windows DLL APIs





//  Start of int_array implementation

/*
  Author          Carlo Hogeveen
  Website         eCarlo.nl/tse
  Version         v1   14 May 2025
  Compatibility   TSE v4 upwards, all variants

  Only use the following functions to manipulate integer arrays:
    integer proc int_array_create(string array_name)
    integer proc int_array_delete(string array_name)
    integer proc int_array_set(string array_name, integer index, integer value)
    integer proc int_array_get(string array_name, integer index)

  int_array_get() returns the value at the index, which defaults to 0.
  All other functions return TRUE or FALSE if they went ok or not.

  An array is intended to only be used in the macro it is created in.
  An array name only needs to be unique within the macro it is created in.
  You can create as many arrays as TSE has free memory and bufferids.
  Creating an array will use up a bufffer id, which limits you to MAXINT
  arrays minus how many bufferids other TSE programs use.
  Array elements can have an index from MININT to MAXINT.
  When an array is created, all array elements exist with value 0.
  Array elements with value 0 do not use memory.
  Setting an array element from not 0 to 0 frees up its memory usage.
  Deleting an array will free up all its memory usage at once.


  NOTE
    I maintain int_array's source as a separate include-file,
    and as an independent and versioned source of truth.

    But for actual use I copy int_array's source as a whole into each macro
    that needs it, so that each such a macro can be tested and distributed
    without int_arrays's version being an outside dependency.

*/


// Do not use or call these int_array_internal_... variables and functions!

integer int_array_internal_id                    = 0
string  int_array_internal_prefix [MAXSTRINGLEN] = 'IntArray'


proc int_array_internal_context()
  integer org_id = 0

  if int_array_internal_id == 0
    int_array_internal_prefix = SplitPath(CurrMacroFilename(), _NAME_) + ':IntArray'
    org_id                    = GetBufferId()
    int_array_internal_id     = CreateTempBuffer()
    ChangeCurrFilename(int_array_internal_prefix, _DONT_PROMPT_|_DONT_EXPAND_|_OVERWRITE_)
    int_array_internal_prefix = int_array_internal_prefix + ':'
    GotoBufferId(org_id)
  endif
end int_array_internal_context


proc int_array_internal_warn(string array_name, string msg)
  if (Query(MsgLevel) in _ALL_MESSAGES_, _WARNINGS_ONLY_)
    if Query(Beep)
      Alarm()
    endif

    MsgBox(int_array_internal_prefix + array_name,
           iif(Val(msg),
               'ERROR ' + msg + '.',
               'ERROR: ' + msg))
  endif
end int_array_internal_warn

// End of int_array_internal_... variables and functions.


// Start of externally callable int_array functions:

integer proc int_array_create(string array_name)
  integer array_id = 0
  integer ok       = FALSE
  integer org_id   = GetBufferId()

  int_array_internal_context()

  array_id = GetBufferInt(int_array_internal_prefix + array_name,
                          int_array_internal_id)
  if array_id
    int_array_internal_warn(array_name, '1')
  else
    array_id = CreateTempBuffer()

    if array_id
      ChangeCurrFilename(int_array_internal_prefix + array_name,
                         _DONT_PROMPT_|_DONT_EXPAND_|_OVERWRITE_)
      if SetBufferInt(int_array_internal_prefix + array_name,
                      array_id,
                      int_array_internal_id)
        ok = TRUE
      else
        int_array_internal_warn(array_name, '2')
      endif
    else
      int_array_internal_warn(array_name, '3')
    endif

    GotoBufferId(org_id)
  endif

  return(ok)
end int_array_create


integer proc int_array_delete(string array_name)
  integer array_id = 0
  integer ok       = FALSE

  int_array_internal_context()
  array_id = GetBufferInt(int_array_internal_prefix + array_name,
                          int_array_internal_id)
  if array_id
    ok = TRUE

    if not AbandonFile(array_id)
      int_array_internal_warn(array_name, '4')
      ok = FALSE
    endif

    if not DelBufferVar(int_array_internal_prefix + array_name,
                        int_array_internal_id)
      int_array_internal_warn(array_name, '5')
      ok = FALSE
    endif
  else
    int_array_internal_warn(array_name, '6')
  endif

  return(ok)
end int_array_delete


integer proc int_array_set(string array_name, integer index, integer value)
  integer array_id = 0
  integer ok       = FALSE

  int_array_internal_context()
  array_id = GetBufferInt(int_array_internal_prefix + array_name,
                          int_array_internal_id)

  if array_id
    if value
      if SetBufferInt(int_array_internal_prefix + array_name + Str(index),
                      value,
                      array_id)
        ok = TRUE
      else
        int_array_internal_warn(array_name, '7')
      endif
    else
      DelBufferVar(int_array_internal_prefix + array_name + Str(index),
                   array_id)
    endif
  else
    int_array_internal_warn(array_name, '8')
  endif

  return(ok)
end int_array_set


integer proc int_array_get(string array_name, integer index)
  integer array_id = 0
  integer value    = 0

  int_array_internal_context()
  array_id = GetBufferInt(int_array_internal_prefix + array_name,
                          int_array_internal_id)

  if array_id
    value = GetBufferInt(int_array_internal_prefix + array_name + Str(index),
                         array_id)
  endif

  return(value)
end int_array_get

// End of int_array implementation





integer proc write_profile_int(string  section_name,
                               string  item_name,
                               integer item_value)
  integer ok = WriteProfileInt(section_name,
                               item_name,
                               item_value)
  if not ok
    Warn(MACRO_NAME; 'error in writing configuration to file "tse.ini".')
    PurgeMacro(MACRO_NAME)
  endif
  return(ok)
end write_profile_int


proc push_on_stack(integer x_pos,
                   integer y_pos,
                   integer old_color,
                   integer new_color)
    g_array_top = g_array_top + 1
    int_array_set('X'       , g_array_top, x_pos)
    int_array_set('Y'       , g_array_top, y_pos)
    int_array_set('OldColor', g_array_top, old_color)
    int_array_set('NewColor', g_array_top, new_color)
end push_on_stack


proc pop_from_stack(var integer x_pos,
                    var integer y_pos,
                    var integer old_color,
                    var integer new_color)
    x_pos       = int_array_get('X'       , g_array_top)
    y_pos       = int_array_get('Y'       , g_array_top)
    old_color   = int_array_get('OldColor', g_array_top)
    new_color   = int_array_get('NewColor', g_array_top)
    g_array_top = g_array_top - 1
end pop_from_stack


proc init_stack()
  g_array_top = 0
end init_stack


integer proc stack_not_empty()
  return(g_array_top > 0)
end stack_not_empty


integer proc invert_color(integer old_color)
  return(~ old_color & 0xFFFFFF)
end invert_color


proc draw_cursor()
  integer char_height_in_pixels = 0
  integer char_width_in_pixels  = 0
  integer cursor_screen_x       = Query(WindowX1) + CurrCol() - 1 - CurrXoffset()
  integer cursor_screen_y       = Query(WindowY1) + CurrRow() - 1
  integer new_color             = 0
  integer old_color             = 0
  integer x_pos                 = 0
  integer y_pos                 = 0

  GetCharWidthHeight(char_width_in_pixels, char_height_in_pixels)
  init_stack()

  for x_pos = (cursor_screen_x - 1) * char_width_in_pixels
            to cursor_screen_x * char_width_in_pixels - 1
    y_pos     = (cursor_screen_y - 1) * char_height_in_pixels
    old_color = GetPixel(g_device_context_handle, x_pos, y_pos)
    new_color = invert_color(old_color)
    SetPixel(g_device_context_handle, x_pos, y_pos, new_color)
    push_on_stack(x_pos, y_pos, old_color, new_color)

    y_pos     = cursor_screen_y * char_height_in_pixels - 1
    old_color = GetPixel(g_device_context_handle, x_pos, y_pos)
    new_color = invert_color(old_color)
    SetPixel(g_device_context_handle, x_pos, y_pos, new_color)
    push_on_stack(x_pos, y_pos, old_color, new_color)
  endfor

  //  y_pos ranges from (hook_pos + 1) to (hook_pos - 1)
  //  to not double-draw the hook points.
  for y_pos = (cursor_screen_y - 1) * char_height_in_pixels + 1
            to cursor_screen_y * char_height_in_pixels - 1  - 1
    x_pos     = (cursor_screen_x - 1) * char_width_in_pixels
    old_color = GetPixel(g_device_context_handle, x_pos, y_pos)
    new_color = invert_color(old_color)
    SetPixel(g_device_context_handle, x_pos, y_pos, new_color)
    push_on_stack(x_pos, y_pos, old_color, new_color)

    x_pos = cursor_screen_x * char_width_in_pixels - 1
    old_color = GetPixel(g_device_context_handle, x_pos, y_pos)
    new_color = invert_color(old_color)
    SetPixel(g_device_context_handle, x_pos, y_pos, new_color)
    push_on_stack(x_pos, y_pos, old_color, new_color)
  endfor
end draw_cursor

proc undraw_cursor()
  integer cur_color             = 0
  integer new_color             = 0
  integer old_color             = 0
  integer x_pos                 = 0
  integer y_pos                 = 0

  while stack_not_empty()
    pop_from_stack(x_pos, y_pos, old_color, new_color)
    cur_color = GetPixel(g_device_context_handle, x_pos, y_pos)
    if cur_color == new_color
      SetPixel(g_device_context_handle, x_pos, y_pos, old_color)
    endif
  endwhile
end undraw_cursor


proc idle()
  g_idle_counts = g_idle_counts + 1
  if g_idle_counts >= 2
    g_idle_counts = 0
    if Query(Cursor)
      // Sound(200, 200)
      Set(Cursor, OFF)
    endif
    if g_box_cursor_state
      undraw_cursor()
    else
      draw_cursor()
    endif
    g_box_cursor_state = not g_box_cursor_state
  endif
end idle


proc post_update_all_windows()
  if QueryEditState() == 0
    draw_cursor()
    g_idle_counts  = MAXINT - 1
    g_box_cursor_state = OFF
  endif
end post_update_all_windows


string proc ff_yyyymmddhhmmss(string fqn)
  integer old_DateFormat = Set(DateFormat, 6)
  integer old_TimeFormat = Set(TimeFormat, 1)
  string result [14] = ''
  if FindThisFile(fqn)
    result = FFDateStr() + FFTimeStr()
  endif
  Set(DateFormat, old_DateFormat)
  Set(TimeFormat, old_TimeFormat)
  return(result)
end ff_yyyymmddhhmmss


proc WhenPurged()
  integer ok = TRUE

  if not EquiStr(MACRO_NAME[Length(MACRO_NAME) - 5: 6], 'Helper')   // (*1)
    int_array_delete('X')
    int_array_delete('Y')
    int_array_delete('OldColor')
    int_array_delete('NewColor')
    ReleaseDC(g_window_handle, g_device_context_handle)

    if not FileExists(MACRO_DPN + 'Helper.mac')
    or CmpiStr(ff_yyyymmddhhmmss(MACRO_DPN + 'Helper.mac'),
               ff_yyyymmddhhmmss(MACRO_DPN +       '.mac')) < 0
      ok = CopyFile(MACRO_DPN + '.mac', MACRO_DPN + 'Helper.mac', TRUE)
    endif

    if  ok
    and FileExists(MACRO_DPN + 'Helper.mac')
      ExecMacro(MACRO_NAME + 'Helper')
    else
      MsgBox(MACRO_NAME + ' ERROR',
             'Could not copy' + Chr(13) +
             '  ' + MACRO_DPN + '.mac' + Chr(13) +
             'to' + Chr(13) +
             '  ' + MACRO_DPN + 'Helper.mac')
    endif
  endif
end WhenPurged


proc WhenLoaded()
  MACRO_NAME              = SplitPath(CurrMacroFilename(), _NAME_)
  MACRO_DPN               = SplitPath(CurrMacroFilename(), _DRIVE_|_PATH_|_NAME_)

  if not EquiStr(MACRO_NAME[Length(MACRO_NAME) - 5: 6], 'Helper')    // (*1)
    g_window_handle         = GetWinHandle()
    g_device_context_handle = GetDC(g_window_handle)

    int_array_create('X')
    int_array_create('Y')
    int_array_create('OldColor')
    int_array_create('NewColor')

    if isGUI()
      Hook(_POST_UPDATE_ALL_WINDOWS_, post_update_all_windows)
      Hook(_IDLE_                   , idle)
      Hook(_ON_ABANDON_EDITOR_      , WhenPurged)
    else
      if  GetProfileInt(MACRO_NAME, 'RepeatConsoleWarning', TRUE)
      and MsgBox(MACRO_NAME + ' abort',
                 Format(MACRO_NAME;
                        'requires the Windows GUI version of TSE.',
                        Chr(13), Chr(13),
                        'Repeat this error message any next time?'),
                  _YES_NO_) == 2
        write_profile_int(MACRO_NAME, 'RepeatConsoleWarning', FALSE)
      endif
      PurgeMacro(MACRO_NAME)
    endif
  endif
end WhenLoaded


proc reenable_cursor()
  Set(Cursor, ON)
  PurgeMacro(MACRO_NAME)
end reenable_cursor


proc Main()
  if EquiStr(MACRO_NAME[Length(MACRO_NAME) - 5: 6], 'Helper')    // (*1)
    Hook(_IDLE_, reenable_cursor)
  endif
end Main

