/*
  Tool            Chcp
  Author          Carlo Hogeveen
  Website         eCarlo.nl/tse
  Compatibility   Windows Console TSE
  Version         v1.1.1   27 Nov 2025 (2)


  This tool changes the current session's code page without leaving TSE.

  If you view a text, and funny characters make you suspect that it was written
  using a different code page than your own default one, then you can try other
  code pages without leaving the text.

  It can switch the session's local code page between:
     437 (US English).
     850 (any other western language and any other English speaking country).
    1252 (the international Windows standard also used in GUI TSE).

  You can check that the code page has changed in TSE menu Util -> Ascii Chart.
  Especially you can compare its bottom characters with those at the bottom of
    https://ecarlo.nl/tse/files/CodePages.html
  Alternatively you can watch these characters change:
      

  If "chcp" cannot change the code page back to your system's default code page
  yet, then let me know if you would like me to add it to this tool.


  KNOWN DEFICIENCIES

    This tool will only ever support 1-byte code pages.
    It will never support multi-byte code pages, like for example the Unicode
    code pages 65001 (UTF-8), 1200 (UTF-16BE), 1201 (UTF-16LE), 12000
    (UTF-32BE), and 12001 (UTF-32LE).

    Using code page 1252 currupts TSE's window borders.
    This is not a macro bug but a TSE bug.
    If you type "chcp 1252" in a new command prompt and start Console TSE from
    there, then you get the exact same TSE bug.


  WARNING
    This tool hacks TSE in a not-supported way.
    There could be unknown side-effects.


  INSTALLATION
    Just compile and execute the macro.

    Optionally add "SetCodePage" to your Potpourri menu, or assign
      ExecMacro('SetCodePage')
    to a key in your .ui file.


  HISTORY

  v1.1.1    27 Nov 2025 (2)
    The one user report of the tool not working might not be a Windows Home
    problem after all. I removed that as a possible cause, but left in an error
    message when the tool fails to modify the code page.

  v1.1      27 Nov 2025
    Found out about its incompatibility with Windows Home edition.
    Now declares its Widows Pro or better requirement, and tests for failure.

  v1        26 Nov 2025
    Initial release.
*/



#ifdef LINUX
  Error: This tool is not compatible with LINUX TSE.
#endif


// Global variable
integer startup_local_code_page = 0


dll "<kernel32.dll>"
  integer proc SetConsoleOutputCP(integer new_cp)

  integer proc GetConsoleOutputCP()
  integer proc GetOEMCP()
  integer proc GetACP()
end


proc WhenPurged()
  if GetConsoleOutputCP() <> startup_local_code_page
    SetConsoleOutputCP(startup_local_code_page)
  endif
end WhenPurged


proc WhenLoaded()
  startup_local_code_page = GetConsoleOutputCP()
  Hook(_ON_ABANDON_EDITOR_, WhenPurged)
end WhenLoaded


proc change_code_page_to(integer new_code_page)
  integer old_code_page = GetConsoleOutputCP()

  if old_code_page <> new_code_page
    SetConsoleOutputCP(new_code_page)
    if GetConsoleOutputCP() <> new_code_page
      Alarm()
      MsgBox(SplitPath(CurrMacroFilename(), _NAME_),
             Format('Changing the code page did not work.'))
    endif
  endif
end change_code_page_to


proc Main()
  if isGUI()
    MsgBox(SplitPath(CurrMacroFilename(), _NAME_),
           'This tool is not compatible with Windows GUI TSE.')
  else
    case GetConsoleOutputCP()
      when 850
        PushKey(<Tab>)
      when 1252
        PushKey(<Tab>)
        PushKey(<Tab>)
    endcase

    case MsgBoxEx(SplitPath(CurrMacroFilename(), _NAME_),
                  Format("Change the session's local code page from"; GetConsoleOutputCP(); 'to ...'),
                  Format('[&437];[&850];[&1252]'))
      when 1
        change_code_page_to(437)
      when 2
        change_code_page_to(850)
      when 3
        change_code_page_to(1252)
    endcase

    // Console TSE needs this trick to update the screen after a code page change.
    ScrollRight()
    UpdateDisplay(_ALL_WINDOWS_REFRESH_)
    ScrollLeft()
    UpdateDisplay(_ALL_WINDOWS_REFRESH_)
  endif
end Main

