/*
  Tool            IndentMatch
  Author          Carlo Hogeveen
  Website         eCarlo.nl/tse
  Compatibility   TSE v4 upwards, all variants
  Version         v1   25 Dec 2025 (3)


  Visually move the cursor to the next/previous line with the same indentation.

  Seeing the text move visually retains understanding it.
  The cursor speed is configurable.
  <Escape> interrupts the moving cursor.

  Note that this tool does not know a programming language's syntax.
  It only finds matching opening and closing syntax if they have the same
  indentation.

  The idea for this tool came from the popular GuideLines extension.
  If we like manually finding a matching indentation using visual guide lines,
  why not also be able to do so automatically?

  Tip:
    There is also my IndentView tool, which provides an alternative way
    to find and understand matching indentations.


  INSTALLATION

    Copy this file to TSE's "mac" directory and compile it there,
    for example by opening it there in TSE and applying the Macro Compile menu.

    Configure the tool's keys either in your .ui file or in the tools's menu.

    For the tool's menu-configured keys to work, the tool's name must be added
    to TSE's Macro AutoLoad List menu, ideally somewhere below the "cuamark"
    macro.


  CONFIGURATION

    Theoretically configuration is optional:
    You can use the tool by executing it as a macro without parameters,
    by then selecting the "Down" or "Up" action in its menu,
    and by then pressing <Enter>.
    This is a clumsy way, but it works.

    In practice you will want to configure keys for the "Down" and "Up" actions
    in order to bypass the menu.

    If you are not knowledgeable about how TSE's key definition precedence
    works, then first read the "KEY DEFINITION PRECEDENCE" section below.

    Configure the tool's keys either in your .ui file or in the tool's menu.

    To configure a key in your .ui file, add these example lines to it
      <key for down action>   ExecMacro("IndentMatch down")
      <Key for up   action>   ExecMacro("IndentMatch up")
    using your own desired keys instead.
    Recompile your .ui file, and restart TSE.

    To configure a key in the tool's menu, execute it without parameters.
    For example type its name in the Macro Execute menu: IndentMatch.

    To configure a key (combination) via the menu, just move to the menu item's
    line and press the desired key (combination).
    In the editor the new key (combination) will be effective immediately.

    Note that the tool must be (auto)loaded for menu-configured keys to work,
    and that the key deefinition must not be overruled by a key definition
    elsewhere.

    You can configure two speeds for moving the cursor to a found match.
    Most of the time the first speed is sufficiant.
    However, when a match is very far away, watching the moving intervening
    text might become tedious and irrelevant, so you can configure the tool to
    use an alternative, faster speed "After ... lines".


  (*1) KEY DEFINITION PRECEDENCE

    If a .ui file configures a key more than once, then the top one is used.

    Keys defined in your .ui file are overruled by keys configured
    in an (auto)loaded macro.

    If multiple (auto)loaded macros configure the same key, then the last
    loaded macro's key is used.
    Initially this is the reverse order of the Macro AutoLoad List menu,
    but that becomes untrue once you manually load macros.
    The Macro Purge menu always lists macros in reverse load order, which makes
    it an accurate determinator of which macro's macro-configured keys will be
    used.


  TODO
    MUST
    SHOULD
    COULD
    WONT


  HISTORY

  v1   25 Dec 2025 (3)
    The first non-development release.

  v0.0.0.6   25 Dec 2025 (2)   Final beta!
    Added a Help option to the tool's menu.

  v0.0.0.5   25 Dec 2025
    Exception for a sequence of consecutive same-indented lines:
    Move to the first or last line of the sequence.

  v0.0.0.4   24 Dec 2025
    Made keys also configurable in the menu.

  v0.0.0.3   23 Dec 2025
    Renamed the tool from MatchIndent to IndentMatch, so that in alphabetical
    lists it appears close to my IndentResize and IndentView tools.
    If added to the Potpourri menu they are all searchable on "Indent".

    Removed the built-in keys.
    For now you need to configure keys in your .ui file.

    Added an action/configuration menu.

    Made the cursor speed configurable.

  v0.0.0.2   17 Dec 2025 (2)
    Fixed the problem that it stopped at empty lines.

  v0.0.0.1   17 Dec 2025
    Initial version.
*/


//  Constants and semi-constants
string  CFG_SECTION   [MAXSTRINGLEN] = ''
#define LONGEST_KEYNAME_LENGTH         37 // "CtrlAltShift KeyPadLeftRightCenterBtn".
#define LONGEST_TWO_WAY_KEYNAME_LENGTH 31 // "CtrlAltShift LeftRightCenterBtn".
string  MACRO_NAME    [MAXSTRINGLEN] = ''
#define MATCH_DIFFERENT_INDENTATION    2
#define MATCH_SAME_INDENTATION         1
#define MATCH_TYPE_UNDETERMINED        0


//  Global variables
integer cfg_after_lines              = 0
integer cfg_initial_speed            = 0
integer cfg_later_speed              = 0
integer cfg_next_key                 = 0
integer cfg_prev_key                 = 0
integer g_ignore_next_key            = FALSE
integer g_stop_main_menu             = FALSE


proc to_beep_or_not_to_beep()
  if Query(Beep)
    Alarm()
  endif
end to_beep_or_not_to_beep


proc profile_error(string section_name,
                   string item_name,
                   string item_value)
  to_beep_or_not_to_beep()
  MsgBox(MACRO_NAME,
         Format('ERROR:'                                      , Chr(13),
                '  Could not write item  "', item_name   , '"', Chr(13),
                '  with value            "', item_value  , '"', Chr(13),
                '  to section            "', section_name, '"', Chr(13),
                '  of configuration file "tse.ini".'))
end profile_error


integer proc write_profile_int(string  section_name,
                               string  item_name,
                               integer item_value)
  integer ok = WriteProfileInt(section_name, item_name, item_value)

  if not ok
    profile_error(section_name, item_name, Str(item_value))
  endif

  return(ok)
end write_profile_int


integer proc remove_profile_item(string section, string item_name)
  integer ok = remove_profile_item(section, item_name)

  if not ok
    to_beep_or_not_to_beep()
    MsgBox(MACRO_NAME, 'Could not remove item "' + item_name +
                       '" from section "'  + section +
                       '" in configuration file "tse.ini".')
  endif

  return(ok)
end remove_profile_item


proc show_help()
  string  full_macro_source_name [MAXSTRINGLEN] = SplitPath(CurrMacroFilename(), _DRIVE_|_PATH_|_NAME_) + '.s'
  string  help_file_name         [MAXSTRINGLEN] = '*** ' + MACRO_NAME + ' Help ***'
  integer hlp_id                                = GetBufferId(help_file_name)
  integer org_id                                = GetBufferId()
  integer tmp_id                                = 0

  if hlp_id
    GotoBufferId(hlp_id)
    UpdateDisplay()
  else
    tmp_id = CreateTempBuffer()
    if LoadBuffer(full_macro_source_name)
      // Separate characters, otherwise my SynCase macro gets confused.
      if lFind('/' + '*', 'g')
        PushBlock()
        UnMarkBlock()
        Right(2)
        MarkChar()
        if not lFind('*' + '/', '')
          EndFile()
        endif
        MarkChar()
        Copy()
        CreateTempBuffer()
        Paste()
        UnMarkBlock()
        PopBlock()
        BegFile()
        ChangeCurrFilename(help_file_name,
                           _DONT_PROMPT_|_DONT_EXPAND_|_OVERWRITE_)
        BufferType(_NORMAL_)
        FileChanged(FALSE)
        BrowseMode(TRUE)
        UpdateDisplay()
      else
        GotoBufferId(org_id)
        Warn('File "', full_macro_source_name, '" has no multi-line comment block.')
      endif
    else
      GotoBufferId(org_id)
      Warn('File "', full_macro_source_name, '" not found.')
    endif
    AbandonFile(tmp_id)
  endif
end show_help


proc indent_match(integer direction)
  integer found                     = FALSE
  integer lines                     = 0
  integer start_PosFirstNonWhite = Max(PosFirstNonWhite(), 1)
  integer match_type                = MATCH_TYPE_UNDETERMINED
  integer speed                     = 0
  integer stop                      = FALSE

  if g_ignore_next_key
    g_ignore_next_key = FALSE
  else
    if direction <> _NONE_
      while CurrPos() < start_PosFirstNonWhite
        Right()
        UpdateDisplay()
      endwhile

      while CurrPos() > start_PosFirstNonWhite
        Left()
        UpdateDisplay()
      endwhile

      repeat
        if direction == _UP_
          Up()
        else
          Down()
        endif

        stop = (CurrLine() in 1, NumLines())

        if  KeyPressed()
        and GetKey() == <Escape>
          stop = TRUE
        endif

        if PosFirstNonWhite()
          if match_type == MATCH_TYPE_UNDETERMINED
            match_type = iif(PosFirstNonWhite() == start_PosFirstNonWhite,
                             MATCH_DIFFERENT_INDENTATION,
                             MATCH_SAME_INDENTATION)
          endif

          if match_type == MATCH_SAME_INDENTATION
            found = (PosFirstNonWhite() <= start_PosFirstNonWhite)
          else
            found = (PosFirstNonWhite() <> start_PosFirstNonWhite)

            if found
              if direction == _UP_
                Down()
              else
                Up()
              endif
            endif
          endif
        endif

        if not (stop or found)
          lines = lines + 1
          speed = iif(lines > cfg_after_lines, cfg_later_speed, cfg_initial_speed)

          if speed <> 9
            UpdateDisplay()
            Delay(8 - speed)
          endif
        endif
      until found
         or stop
    endif
  endif
end indent_match


integer proc ReadNum1(integer n)
  string s[1] = Str(n)
  return (iif(ReadNumeric(s), Val(s), n))
end ReadNum1


integer proc ReadNum9(integer n)
  string s[9] = Str(n)
  return (iif(ReadNumeric(s), Val(s), n))
end ReadNum9


proc set_initial_speed()
  cfg_initial_speed = ReadNum1(cfg_initial_speed)
  cfg_initial_speed = Max(cfg_initial_speed, 1)
  cfg_initial_speed = Min(cfg_initial_speed, 9)
  write_profile_int(CFG_SECTION, 'InitialSpeed', cfg_initial_speed)
end set_initial_speed


proc set_later_speed()
  cfg_later_speed = ReadNum1(cfg_later_speed)
  cfg_later_speed = Max(cfg_later_speed, 1)
  cfg_later_speed = Min(cfg_later_speed, 9)
  write_profile_int(CFG_SECTION, 'LaterSpeed', cfg_later_speed)
end set_later_speed


proc set_after_lines()
  cfg_after_lines = ReadNum9(cfg_after_lines)
  cfg_after_lines = Min(cfg_after_lines, 999999999)
  cfg_after_lines = Max(cfg_after_lines,         0)
  write_profile_int(CFG_SECTION, 'AfterLines', cfg_after_lines)
end set_after_lines


string proc get_next_key()
  return(iif(cfg_next_key, KeyName(cfg_next_key), ''))
end get_next_key


string proc get_prev_key()
  return(iif(cfg_prev_key, KeyName(cfg_prev_key), ''))
end get_prev_key


proc set_if_new_key_for_menu_item(    integer menu_line,
                                  var integer cfg_key_code,
                                      string  cfg_key_name)
  string  attr1         [1] = ''
  string  dummy         [1] = ''
  integer old_cfg_key_code  = cfg_key_code
  integer pressed_key_code  = Query(Key)

  GetStrAttrXY(1, menu_line, dummy, attr1, 1)

  if Asc(attr1) == Query(MenuSelectAttr)

    // Ignore single characters, navigation keys, and control keys except <Del>.
    if  Length(KeyName(pressed_key_code)) <> 1
    and not (pressed_key_code in <Alt>,
                                 <CapsLock>,
                                 <CtrlAltShift LeftRightCenterBtn>,
                                 <CursorDown>,
                                 <CursorLeft>,
                                 <CursorRight>,
                                 <CursorUp>,
                                 <end>,
                                 <Enter>,
                                 <Escape>,
                                 <GreyCursorDown>,
                                 <GreyCursorLeft>,
                                 <GreyCursorRight>,
                                 <GreyCursorUp>,
                                 <GreyEnd>,
                                 <GreyEnter>,
                                 <GreyHome>,
                                 <GreyIns>,
                                 <GreyPgDn>,
                                 <GreyPgUp>,
                                 <Home>,
                                 <Ins>,
                                 <LeftBtn>,
                                 <NumLock>,
                                 <PgDn>,
                                 <PgUp>,
                                 <RightBtn>,
                                 <SpaceBar>,
                                 <Tab>,
                                 <WheelDown>,
                                 <WheelUp>)

      if (pressed_key_code in <Del>, <GreyDel>)
        cfg_key_code = 0
      else
        cfg_key_code = pressed_key_code
      endif

      if cfg_key_code <> old_cfg_key_code
        if cfg_key_code
          write_profile_int(CFG_SECTION, cfg_key_name, cfg_key_code)
        else
          RemoveProfileItem(CFG_SECTION, cfg_key_name)
        endif
      endif

      Set(Key, -1)
      PushKey(<Enter>)
      g_ignore_next_key = TRUE
      g_stop_main_menu  = FALSE
    endif
  endif
end set_if_new_key_for_menu_item


proc after_getkey_configure_menu_key()
  //  set_if_new_key_for_menu_item()'s first parameter is the menu line.
  //
  //  Counting menu lines starts at the line below the menu title and includes
  //  any dividing lines and empty lines.
  //
  //  If that menu line has the Query(MenuSelectAttr) color, i.e. the line is
  //  selected, then it is checked whether the last key is an allowed
  //  configuration Key.
  //  If so, then the key is set using the other parameters.
  //
  //  Note: It seems that inside this hook the TSE debugger still steps thru
  //  code with F7, but ignores keys to view variable values.

  set_if_new_key_for_menu_item(1, cfg_next_key, 'FindNextMatchKey')
  set_if_new_key_for_menu_item(2, cfg_prev_key, 'FindPreviousMatchKey')
end after_getkey_configure_menu_key


proc after_getkey_do()
  case Query(Key)
    when cfg_next_key
      Set(Key, -1)
      indent_match(_DOWN_)
    when cfg_prev_key
      Set(Key, -1)
      indent_match(_UP_)
  endcase
end after_getkey_do


menu main_menu()
  history

  '&Down'
    [get_next_key(): LONGEST_TWO_WAY_KEYNAME_LENGTH],
    indent_match(_DOWN_),
    _MF_CLOSE_ALL_BEFORE_|_MF_ENABLED_,
    '<Enter> finds next match. Key(combo) sets key. <Del> deletes key.'

  '&Up'
    [get_prev_key(): LONGEST_TWO_WAY_KEYNAME_LENGTH],
    indent_match(_UP_),
    _MF_CLOSE_ALL_BEFORE_|_MF_ENABLED_,
    '<Enter> finds previous match. Key(combo) sets key. <Del> deletes key.'

  '&Initial cursor speed'
    [cfg_initial_speed:1],
    set_initial_speed(),
    _MF_ENABLED_|_MF_DONT_CLOSE_,
    'Speeds 1 - 8 visually move the cursor, speed 9 jumps.'

  '&Later cursor speed'
    [cfg_later_speed:1],
    set_later_speed(),
    _MF_ENABLED_|_MF_DONT_CLOSE_,
    'Speeds 1 - 8 visually move the cursor, speed 9 jumps.'

  '&After ... lines'
    [cfg_after_lines:9],
    set_after_lines(),
    _MF_ENABLED_|_MF_DONT_CLOSE_,
    'After how many lines change the initial cursor speed to the later one.'

  '&Help',
    show_help(),
    _MF_CLOSE_ALL_BEFORE_|_MF_ENABLED_,
    'Show the documentation.'
end main_menu


proc do_main_menu()
  Hook(_AFTER_GETKEY_, after_getkey_configure_menu_key)

  repeat
    g_stop_main_menu = TRUE
    main_menu(MACRO_NAME)
  until g_stop_main_menu
     or MenuOption() == 0

  UnHook(after_getkey_configure_menu_key)
end do_main_menu


proc WhenPurged()
end WhenPurged


proc WhenLoaded()
  MACRO_NAME  = SplitPath(CurrMacroFilename(), _NAME_)
  CFG_SECTION = MACRO_NAME + ':Configuration'

  cfg_initial_speed = GetProfileInt(CFG_SECTION, 'InitialSpeed', 7)
  cfg_later_speed   = GetProfileInt(CFG_SECTION, 'LaterSpeed', 8)
  cfg_after_lines   = GetProfileInt(CFG_SECTION, 'AfterLines', Query(WindowRows) * 2)
  cfg_next_key      = GetProfileInt(CFG_SECTION, 'FindNextMatchKey', 0)
  cfg_prev_key      = GetProfileInt(CFG_SECTION, 'FindPreviousMatchKey', 0)

  Hook(_AFTER_GETKEY_, after_getkey_do)
end WhenLoaded


proc Main()
  case Lower(Query(MacroCmdLine))
    when 'down'
      indent_match(_DOWN_)
    when 'up'
      indent_match(_UP_)
    when ''
      do_main_menu()
    otherwise
      MsgBox(MACRO_NAME, 'Illegal parameter "' + Query(MacroCmdLine) + '"')
  endcase

  if  cfg_next_key == 0
  and cfg_prev_key == 0
    PurgeMacro(MACRO_NAME)
  endif
end Main


