/*
  Macro           ListInstalledVersions
  Author          Carlo Hogeveen
  Website         eCarlo.nl/tse
  Compatibility   Windows TSE v4.41.35 upwards
  Version         v1   5 Feb 2026

  This tool lists all installed Windows 32-bit TSE versions in a drive or path.

  The tool runs in Windows TSE v4.41.35 upwards.
  It lists installations from TSE 2.8 upwards.
  For each found installation it lists lots of version information.
  It asks for the drive or path to search in.


  For the average user it might be interesting to check if and where they have
  TSE installations on a particular drive, and what versions these are.


  For me as a macro programmer it is necessary to know which external TSE
  versions in TSE's read.me file match with which internal TSE versions.

  And luckily I already keep lots of old TSE installations, both for
  compatibility testing and to determine since which version a new bug occurs.

  The relevance of TSE's internal versions is this:
  For macro programmers the macro language has an INTERNAL_VERSION compilation
  directive as of TSE v4.41.44 (18 Jan 2021).
  With this compilation directive a macro can use a new macro laguage feature
  in those TSE versions that support it, and provide a work-around or a
  graceful degradation in those TSE versions that do not support the feature.


  INSTALLATION AND USAGE

    Just compile the macro, execute it, and provide a drive or path when asked.

    The default is to list the full path of a found installation, but for
    sharing the result list with others you can select to only list each
    installation's parent directory.

    The result list of TSE installations is sorted on their internal versions.

    When running the tool, ignore and <Escape> warnings and errors:
      Ideally you will not get any warnings and errors.
      Lots of care was taken to avoid them.
      However, the tool does need to run the g32.exe of the found TSE
      installations, or their e32.exe if g32.exe does not exist.
      Autoloaded macros, history, and some configuration are avoided by copying
      the executable to an empty temporary directory and running it from there.
      For me a few installations still manage to pop-up a message or warning,
      which in my cases turn out to be ignorable and <Escape>-able,
      which then does not interfere with generating the list.

    Disclaimer 1:
      TSE is very programmable, and it is therefore impossible to anticipate
      every personal modification a user might have made, especially
      modifications that are burned into an executable itself, as TSE allows.

      This tool runs OK on my computer, and will probably run OK on most
      people's computers, but there is no garantee that it will.

    Disclaimer 2:
      The listed information might be wrong!
      The tool just lists the information it finds:
      - The user might have named the parent directory incorrectly.
      - Semware has on occasion set version information incorrectly and/or
        inconsistently.
      Often the tool shows so many versions of an installation, that we can
      figure out if and where an error occurred.



  HOW TSE VERSIONS WORK

    "INTERNAL" TSE VERSIONS

      TSE's macro language can use 3 versions:
        Version()         A run-time function.  Exists since forever.
        EDITOR_VERSION    A compiler directive. Exists since TSE v3.
        INTERNAL_VERSION  A compiler directive. Exists since TSE v4.41.44.

      Version()
        This run-time function returns a simple incremental integer.
        You cannot derive this integer from TSE's external version or the other
        way around: Those are not related.

      EDITOR_VERSION
        This compiler directive only ever had a few values:
          Release   EDITOR_VERSION
            v2.5      0x2500
            v2.8      0x2800
            v3.0      0x3000
            v4.0      0x4000
            v4.2      0x4200
            v4.4      0x4400
          > v4.4      0x4500    (All beta and freeware releases after v4.4)

      INTERNAL_VERSION
        This compiler directive returns the same simple incremented integer
        as the run-time function Version() does.

        In theory INTERNAL_VERSION only needs to change when a release's macro
        language changes, and thus when its compiler changes. That method was
        used in the beginning, but it lead to human error maintaining it.
        In practice INTERNAL_VERSION nowadays has a new value for every new TSE
        release.



    EXTERNAL TSE VERSIONS

      External TSE versions are a mess.
      They are inconsistent, sometimes wrong, and sometimes absent.

      To compensate for that, this tool queries and lists multiple external
      version sources in each TSE installation, and lets the user try to make
      sense of it.

      When available the tool lists each installation's:
      - Full path or only its parent directory.
      - The editor macro language's:
        - EDITOR_VERSION   compiler directive or 0.
        - INTERNAL_VERSION compiler directive or 0.
        - The Version() function result,
          which is also the internal version, even for old editors.
      - All the editor Help -> About() window's version-related strings.
        TSE's ABout() function exists since TSE v4.
      - The version a.k.a. the first line from the running the compiler.
      - The top version from the read.me file,
        which only exists in the beta and freeware versions after TSE v4.4.

      The tool sorts the list of TSE installations on their internal versions.
      Again, it can be correct that some TSE installations with different
      external versions have the same internal version.


  TODO
    MUST
    SHOULD
    COULD
    WONT


  HISTORY

  v1   5 Feb 2026
    Initial release.

*/



/*
  T E C H N I C A L   B A C K G R O U N D   I N F O


  TWO MACROS!

    This source file contains 2 macros:
    - The main one that runs after compiling the macro normally.
    - A helper macro that the main macro creates by compiling the source file
      with the compiler directive HELPER_MACRO.
      For each found TSE installation the helper macro is compiled in a
      temporary directory.

    For testing and debugging the helper macro's source-part can be copied from
    the combined source into a stand-alone macro.
    In the copy the included DEBUG compiler directive must be set to TRUE.
    The copy can be compiled normally by almost all TSE versions,
    and be run from there for testing and debugging.
*/





#ifdef HELPER_MACRO

  /*
    lVersion()
      This function pops Up a an old-type Warn() line in TSE < v4,
      and pops Up 2(!) windows in the current Version.
      This tool does not use the lVersion() function.
    Version()
      Returns internal version as an integer from at least TSE v2.8 upwards.
    About()
      Pops up a window that can contain various version information.
      What version info About() shows varies by TSE version.
      This tool gathers such version info as best it can.
      The About() function exists from TSE v4 upwards.
  */

  #define DEBUG FALSE


  //  EDITOR_VERSION exists since TSE v3.0.
  #ifndef EDITOR_VERSION
    #define EDITOR_VERSION 0
  #endif

  //  INTERNAL_VERSION exists since TSE v4.41.44.
  #ifndef INTERNAL_VERSION
    #define INTERNAL_VERSION 0
  #endif


  #if EDITOR_VERSION >= 4000h
    string about_versions [MAXSTRINGLEN] = ''

    proc after_getkey()
      string  about_external_version [MAXSTRINGLEN] = ''
      string  about_internal_version [MAXSTRINGLEN] = ''
      string  attrs                  [MAXSTRINGLEN] = ''
      string  chars                  [MAXSTRINGLEN] = ''
      integer video_win_y                           = 0

      UnHook(after_getkey)

      for video_win_y = 1 to Query(PopWinRows)
        GetStrAttrXY(1, video_win_y, chars, attrs, Query(PopWinCols))
        if video_win_y == 1
          about_external_version = Trim(chars) + '.'
        elseif Pos('internal', Lower(chars))
          if about_internal_version == ''
            about_internal_version = Trim(chars) + '.'
          else
            about_internal_version = about_internal_version + ' ' +
                                     Trim(chars) + '.'
          endif
        elseif Pos('version', Lower(chars))
          about_external_version   = about_external_version + ' ' +
                                     Trim(chars) + '.'
        endif
      endfor

      about_versions = Trim(about_external_version + ' ' +
                            about_internal_version)
      ChainCmd()
    end after_getkey


    proc get_about_versions()
      Hook(_AFTER_GETKEY_, after_getkey)
      PushKey(<Escape>)
      About()
    end get_about_versions
  #endif


  #if not DEBUG
    proc disable_this_hook()
      BreakHookChain()
    end
  #endif


  proc Main()
    #if DEBUG
      NewFile()
    #else
      CreateTempBuffer()
    #endif

    AddLine(Format(Str(EDITOR_VERSION, 16), Chr(3),
                   INTERNAL_VERSION       , Chr(3),
                   Version()              , Chr(3)))

    #if EDITOR_VERSION >= 4000h
      get_about_versions()
      EndLine()
      InsertText(about_versions)
    #endif

    SaveAs(SplitPath(CurrMacroFilename(), _DRIVE_|_PATH_|_NAME_) + '.tmp',
           _DONT_PROMPT_|_OVERWRITE_)

    // Exit TSE session without leaving a trace:
    Set(PersistentHistory    , OFF) // Disable saving session's history.
    Set(PersistentRecentFiles, OFF) // Disable saving session's history.
    #if not DEBUG
      Hook(_ON_ABANDON_EDITOR_, disable_this_hook)
      Hook(_ON_EXIT_CALLED_   , disable_this_hook)
      PushKey(<Escape>) // Exit (re)start-menu.
      AbandonEditor()
    #endif
  end Main


#else




  //  Constants and semi-constants
  #define ANY_ATTRIBUTES              -1
  string  HELPER_DPN [MAXSTRINGLEN] = ''
  #define INVALID_HANDLE              -1
  #define LINK_ATTRIBUTE              1024 // No _link_ in < TSE 4.50rc19.
  string  LOG_FQN    [MAXSTRINGLEN] = ''
  string  MACRO_FQN  [MAXSTRINGLEN] = ''
  string  MACRO_NAME [MAXSTRINGLEN] = ''
  string  TMP_PATH   [MAXSTRINGLEN] = ''


  //  Global variables
  integer g_list_type = 0


  proc ch_dir(string path)
    LogDrive(SubStr(SplitPath(path, _DRIVE_), 1, 1))
    ChDir(path)
  end ch_dir



  proc delete_content_in_path(string path)
    integer handle = FindFirstFile(AddTrailingSlash(path) + '*', ANY_ATTRIBUTES)
    if handle <> INVALID_HANDLE
      repeat
        if FFAttribute() & _DIRECTORY_
          if not (FFName() in '.', '..')
            delete_content_in_path(AddTrailingSlash(path) + FFName())
            RmDir(AddTrailingSlash(path) + FFName())
          endif
        else
          EraseDiskFile(AddTrailingSlash(path) + FFName())
        endif
      until not FindNextFile(handle, ANY_ATTRIBUTES)
      FindFileClose(handle)
    endif
  end delete_content_in_path


  proc list_tse_installation(string path)
    string  cmd              [MAXSTRINGLEN] = ''
    string  compiler_version [MAXSTRINGLEN] = ''
    string  dir              [MAXSTRINGLEN] = SplitPath(path, _NAME_|_EXT_)
    string  editor_versions  [MAXSTRINGLEN] = Chr(3) + Chr(3) + Chr(3)
    integer log_id                          = 0
    integer lst_id                          = 0
    string  readme_version   [MAXSTRINGLEN] = ''
    integer tmp_id                          = 0
    string  org_editor_fqn   [MAXSTRINGLEN] = ''
    string  tmp_editor_fqn   [MAXSTRINGLEN] = ''

    ch_dir(TMP_PATH)

    if     FileExists( AddTrailingSlash(path) + 'g32.exe')
      org_editor_fqn = AddTrailingSlash(path) + 'g32.exe'
      tmp_editor_fqn = AddTrailingSlash('.')  + 'g32.exe'
    elseif FileExists( AddTrailingSlash(path) + 'e32.exe')
      org_editor_fqn = AddTrailingSlash(path) + 'e32.exe'
      tmp_editor_fqn = AddTrailingSlash('.')  + 'e32.exe'
    else
      org_editor_fqn = ''
      tmp_editor_fqn = ''
    endif

    KeyPressed()
    Message(iif(org_editor_fqn == '', path, org_editor_fqn))
    KeyPressed()

    if  org_editor_fqn <> ''
    and CopyFile(MACRO_FQN, HELPER_DPN + '.s')
    and FileExists(HELPER_DPN + '.s')
      cmd = Format(QuotePath(AddTrailingSlash(path) + 'sc32.exe');
                   '-dHELPER_MACRO';
                   QuotePath(HELPER_DPN + '.s'); '>'; LOG_FQN; '2>&1')
      Dos(cmd, _DONT_PROMPT_|_DONT_CLEAR_|_RETURN_CODE_|_START_HIDDEN_)
    endif

    if FileExists(LOG_FQN)
      lst_id = GetBufferId()
      log_id = EditBuffer(LOG_FQN, _SYSTEM_)
      if log_id
        compiler_version = Trim(GetText(1, MAXSTRINGLEN))
      endif
      GotoBufferId(lst_id)
      AbandonFile(log_id)
    endif

    if FileExists(HELPER_DPN + '.mac')
      //  Copy the found editor to the temporary directory where it has no
      //  autoloaded macros, history, or other non-standard settings.
      CopyFile(org_editor_fqn, tmp_editor_fqn)

      if FileExists(tmp_editor_fqn)
        cmd = Format(QuotePath(tmp_editor_fqn);
                     '-e',
                     QuotePath(HELPER_DPN + '.mac'))
        Dos(cmd, _DONT_PROMPT_|_DONT_CLEAR_|_RETURN_CODE_|_START_HIDDEN_)
      endif
    endif

    //  Account for the unlikely possibility, that the started editor changed
    //  the current path away from TMP_PATH.
    ch_dir(TMP_PATH)

    if FileExists(HELPER_DPN + '.tmp')
      lst_id = GetBufferId()
      tmp_id = EditBuffer(HELPER_DPN + '.tmp', _SYSTEM_)
      if tmp_id
        editor_versions = Trim(GetText(1, MAXSTRINGLEN))
      endif
      GotoBufferId(lst_id)
      AbandonFile(tmp_id)
    endif

    if  org_editor_fqn <> ''
    and FileExists(AddTrailingSlash(path) + 'read.me')
      lst_id = GetBufferId()
      tmp_id = EditBuffer(AddTrailingSlash(path) + 'read.me', _SYSTEM_)
      if tmp_id
        while readme_version == ''
        and   lFind('--------------------', '^')
          Down()
          if lFind('20[0-9][0-9]', 'cgx')
          or lFind('v4\.'        , 'cgx')
            Down()
            if lFind('--------------------', '^cg')
              Up()
              readme_version = Trim(GetText(1, MAXSTRINGLEN))
              Down()
            endif
          endif
          EndLine()
        endwhile
        GotoBufferId(lst_id)
        AbandonFile(tmp_id)
      endif
    endif

    if org_editor_fqn <> ''
      AddLine(iif(g_list_type == 2, dir, path))
      EndLine()
      InsertText(Chr(3))
      InsertText(editor_versions)
      InsertText(Chr(3))
      InsertText(compiler_version)
      InsertText(Chr(3))
      InsertText(readme_version)
      BegLine()
    endif

    KeyPressed()
    UpdateDisplay(_ALL_WINDOWS_REFRESH_)
    KeyPressed()

    delete_content_in_path(TMP_PATH)
  end list_tse_installation


  proc search_path(string path)
    integer handle = INVALID_HANDLE

    KeyPressed()
    Message(path)
    KeyPressed()

    if FindThisFile(AddTrailingSlash(path) + 'sc32.exe')
      if Lower(SubStr(SplitPath(path, _NAME_|_EXT_), 1, 3))          <> 'bak'
      or Val(SubStr(SplitPath(path, _NAME_|_EXT_), 4, MAXSTRINGLEN)) == 0
        list_tse_installation(path)
      endif
    endif

    handle = FindFirstFile(AddTrailingSlash(path) + '*', ANY_ATTRIBUTES)
    if handle <> INVALID_HANDLE
      repeat
        if       FFAttribute() & _DIRECTORY_
        and not (FFName() in '.', '..')
        and not (FFAttribute() & LINK_ATTRIBUTE)  //  Avoid circular links.
          search_path(AddTrailingSlash(path) + FFName())
        endif
      until not FindNextFile(handle, ANY_ATTRIBUTES)
      FindFileClose(handle)
    endif
  end search_path


  integer proc macro_source_exists()
    if not FileExists(MACRO_FQN)
      MsgBox(MACRO_NAME,
             Format("This macro's source file does not exist:", Chr(13);;
                    MACRO_FQN))
      return(FALSE)
    endif

    return(TRUE)
  end macro_source_exists


  integer proc get_list_type()
    g_list_type = MsgBoxEx(MACRO_NAME,
                           "List TSE installations' full path or only their parent directory?",
                           '[full &Path];[only parent &Directory]')
    return(g_list_type)
  end get_list_type


  proc format_fields_into_columns(string separator)
    integer field_index                = 0
    integer field_justification        = 0
    integer field_pos                  = 0
    string  field_value [MAXSTRINGLEN] = ''
    integer old_Insert                 = Set(Insert, ON)

    BegFile()
    repeat
      field_index = 0
      field_pos   = 1
      while NumTokens(GetText(field_pos, MAXSTRINGLEN), separator)
        field_index  = field_index + 1
        field_value  = GetToken(GetText(field_pos, MAXSTRINGLEN), separator, 1)
        if Length(field_value) > GetBufferInt(Str(field_index))
          SetBufferInt(Str(field_index), Length(field_value))
        endif
        field_pos    = field_pos + Length(field_value) + 1
      endwhile
    until not Down()

    BegFile()
    repeat
      field_index = 0
      field_pos   = 1
      while NumTokens(GetText(field_pos, MAXSTRINGLEN), separator)
        field_index  = field_index + 1
        field_value  = GetToken(GetText(field_pos, MAXSTRINGLEN), separator, 1)
        if Length(field_value) < GetBufferInt(Str(field_index))
          GotoPos(field_pos)
          if Length(field_value) == 0
            InsertText(Format('': GetBufferInt(Str(field_index))))
          else
            field_justification = iif((field_index in 2, 3, 4), 1, -1)
            DelChar(Length(field_value))
            InsertText(Format(field_value:
                                GetBufferInt(Str(field_index)) *
                                  field_justification))
          endif
        endif
        field_pos = field_pos + GetBufferInt(Str(field_index))
        GotoPos(field_pos)
        if GetText(field_pos, 1) == separator
          DelChar()
          InsertText(' | ')
          field_pos = field_pos + 3
        endif
      endwhile
    until not Down()

    Set(Insert, old_Insert)
  end format_fields_into_columns


  proc WhenLoaded()
    MACRO_NAME = SplitPath(CurrMacroFilename(), _NAME_)
    MACRO_FQN  = SplitPath(CurrMacroFilename(), _DRIVE_|_PATH_|_NAME_) + '.s'

    TMP_PATH   = MakeTempName(GetTempPath(), 'tse')

    HELPER_DPN = AddTrailingSlash('.') + MACRO_NAME + '_Helper'
    LOG_FQN    = HELPER_DPN + '.log'
  end WhenLoaded


  proc WhenPurged()
  end WhenPurged


  proc Main()
    integer sort_col_from           = 0
    integer sort_col_to             = 0
    string  old_path [MAXSTRINGLEN] = CurrDir()
    string  top_path [MAXSTRINGLEN] = ''

    if FileExists(TMP_PATH)
      MsgBox(MACRO_NAME,
             Format('Error: Temporary path already exists:', Chr(13);;
                    QuotePath(TMP_PATH)))
      PurgeMacro(CurrMacroFilename())
      return()
    endif

    MkDir(TMP_PATH)
    if not (FileExists(TMP_PATH) & _DIRECTORY_)
      MsgBox(MACRO_NAME,
             Format('Error: Could not create temporary path:', Chr(13);;
                    QuotePath(TMP_PATH)))
      PurgeMacro(CurrMacroFilename())
      return()
    endif

    ch_dir(TMP_PATH)
    if not EquiStr(CurrDir(), AddTrailingSlash(TMP_PATH))
      MsgBox(MACRO_NAME,
             Format('Error: Could not change current path to:', Chr(13);;
                    QuotePath(TMP_PATH)))
      PurgeMacro(CurrMacroFilename())
      return()
    endif

    if  macro_source_exists()
    and Ask('Top directory to search for old TSE installations:',
            top_path,
            GetFreeHistory(MACRO_NAME + ':TopSearchDir'))
    and FileExists(top_path) & _DIRECTORY_
    and get_list_type()
      NewFile()
      top_path = RemoveTrailingSlash(top_path)
      search_path(top_path)

      BegFile()
      InsertLine(Format(Chr(3), 'EDITOR_', Chr(3), 'INTERNAL', Chr(3), Chr(3),
                        Chr(3), Chr(3)))
      AddLine(Format(iif(g_list_type == 2, 'DIR', 'PATH'), Chr(3),
                     'VERSION', Chr(3), '_VERSION', Chr(3),
                     'Version()', Chr(3), 'EDITOR-VERSION-STRINGS', Chr(3),
                     'COMPILER-VERSION-STRING', Chr(3), 'README_VERSION_STRING'))
      format_fields_into_columns(Chr(3))

      BegFile()
      if NumLines() >= 3
        GotoLine(3)
        if  lFind('|', 'cg')
        and lFind('|', 'c+')
        and lFind('|', 'c+')
          sort_col_from = CurrCol() + 2
          if lFind('|', 'c+')
            sort_col_to  = CurrCol() - 2
            if sort_col_from < sort_col_to
              MarkColumn(3, sort_col_from, NumLines(), sort_col_to)
              ExecMacro('sort')
              UnMarkBlock()
            endif
          endif
        endif
      endif
    endif

    GotoLine(2)
    AddLine(Format('': Min(255, LongestLineInBuffer()): '-'))
    BegLine()
    while CurrLineLen() < LongestLineInBuffer()
      InsertText('-', _INSERT_)
    endwhile
    BegFile()

    ch_dir(old_path)
    delete_content_in_path(TMP_PATH)
    RmDir(TMP_PATH)

    PurgeMacro(CurrMacroFilename())
  end Main
#endif


